/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/

package cnrg.itx.signal;

import java.io.*;
import java.net.*;
import cnrg.itx.signal.SignalID;  
import cnrg.itx.ds.UserID;  

/**
 * This is the base class representing the Signaling Packet that is sent to and from all Signaling components.
 * All signal packets inherit from this class and add specific information.
 */
public class SigPacket extends Object implements Serializable
{
	/** The method this packet represents **/
	protected int currentMethod = -1;
	/** The type this packet represents **/
	protected int currentPacket = -1;
	/** UserID of the destination **/
	protected UserID myDestination = null;
	/** Any custom object being sent by an application  **/
	protected Object custom = null;
	
	/**
	 * Default Constructor constructs an invalid SigPacket
	 * 
	 * @param  None.
	 * @return void
	 */
	public SigPacket(){
		this(-1,-1,null);
	}
	
	/**
	 * Overloaded Constructor constructs a SigPacket that is to be used for a specific method
	 * and packet type.
	 * 
	 * @param   m is an integer code for methods handled by Signaling.
	 * @param   p is an integer code for packet type
	 * @return  void
	 */	
	public SigPacket(int m, int p){
		this(m, p, null);
	}

	/**
	 * Overloaded Constructor constructs a SigPacket that is to be used for a specific method,
	 * packet type and Custom Object
	 * 
	 * @param   m is an integer code for methods handled by Signaling.
	 * @param   p is an integer code for packet type
	 * @param   o is any custom object an application may want to send to a peer.
	 * @return  void
	 */	
	public SigPacket(int m, int p, Object o){
		currentMethod = m;
		currentPacket = p;
		custom = o;
	}
	
	/**
	 * Returns the method ID associated with this packet
	 * 
	 * @param	None.
	 * @return	integer value representing the method ID
	 */
	public int getMethodID(){
		return currentMethod;
	}

	/**
	 * Returns the type of the Sigpacket
	 * 
	 * @param   None.
	 * @return  integer value representing the packet type
	 */
	public int getPacketID(){
		return currentPacket;
	}

	/**
	 * Sets the packet type.
	 * 
	 * @param   pt is the integer value of the packet type
	 * @return  void
	 */	
	public void setPacketType(int pt){
		currentPacket= pt;
	}

	/**
	 * Sets the method type.
	 * 
	 * @param   mt is the integer value of the method type
	 * @return  void
	 */	
	public void setMethodType(int mt){
		currentMethod = mt;
	}
	
	/**
	 * Sets the custom object.
	 * 
	 * @param   o is any object that is being sent in the SigPacket
	 * @return  void
	 */	
	public void setCustomObject(Object o){
		custom = o;
	}
	
	/**
	 * Returns the custom object contained in the SigPacket
	 * 
	 * @param   None.
	 * @return  integer value representing the packet type
	 */
	public Object getCustomObject(){
		return custom;
	}

	/**
	 * Sets the UserID of the Destination.
	 * 
	 * @param   uid is the UserID object representing the Destination
	 * @return  void
	 */	
	public void setDestination(UserID uid){
		myDestination = uid;
	}

	/**
	 * Returns the UserID of the destination Signaling component
	 * 
	 * @param   None.
	 * @return  integer value representing the packet type
	 */	
	public UserID getDestination(){
		return myDestination;
	}

	/** This method informs the callee whether the packet is for the Dial method.
	 * 
	 * @param   None
	 * @return  TRUE if it is the Dial method, FALSE otherwise
	 */
	public boolean isDial(){
		return (currentMethod == SignalID.DIAL);
	}

	/** This method informs the callee whether the packet is for the Hangup method.
	 * 
	 * @param   None
	 * @return  TRUE if it is the Hangup method, FALSE otherwise
	 */	
	public boolean isHangup(){
		return (currentMethod == SignalID.HANGUP);
	}
	
	/** This method informs the callee whether the packet is for the SendDTMF method.
	 * 
	 * @param   None
	 * @return  TRUE if it is the SendDTMF method, FALSE otherwise
	 */
	public boolean isSendDTMF(){
		return (currentMethod == SignalID.SENDDTMF);
	}
	
	/** This method informs the callee whether the packet is for the AliveQuery method.
	 * 
	 * @param   None
	 * @return  TRUE if it is the AliveQuery method, FALSE otherwise
	 */
	public boolean isAliveQuery(){
		return (currentMethod == SignalID.ALIVEQUERY);
	}
	
	/** This method informs the callee whether it is an Invite Packet
	 * 
	 * @param   None
	 * @return  TRUE if it is an Invite Packet, FALSE otherwise
	 */
	public boolean isInvitePacket(){
		return (currentPacket == SignalID.INVITE);
	}
	
	/** This method informs the callee whether it is a Result Packet
	 * 
	 * @param   None
	 * @return  TRUE if it is a Result Packet, FALSE otherwise
	 */
	public boolean isResultPacket(){
		return (currentPacket == SignalID.RESULT);
	}
	
	/** This method informs the callee whether it is a Confirm Packet
	 * 
	 * @param   None
	 * @return  TRUE if it is an Confirm Packet, FALSE otherwise
	 */
	public boolean isConfirmPacket(){
		return (currentPacket == SignalID.CONFIRM);
	}
	
	/** This method informs the callee whether a custom object is contained within this
	 *  packet or not.
	 * 
	 * @param   None
	 * @return  TRUE if a custom object is embedded in the packet, FALSE otherwise
	 * 
	 */
	public boolean hasCustomObject(){
		return (custom == null);
	}
}
